%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: apply_drift_corr_conc
%
% Scope:    apply drift correction to concentrations
%
% Input:    data (structure),
%           bindata (structure),
%           binref (structure),
%           anchor gas name (string),
%           list of gas names (cell),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           drift corr index (species) (double),
%           index plot requested (logical)
%
% Output:   corrected "bindata" and "binref" structures,
%           overview figures (if requested)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ref, binref, bindata] = apply_drift_corr_conc(data, bindata, binref, anchor_gas, gasnames, gastypeflag, drift_idx_spec, plot_requested)
    
    tic;fprintf('Running function apply_drift_corr_conc...\n');
        
    target_idx=[];
    if drift_idx_spec(1,1)==1
        target_idx(1,width(target_idx)+1)=2;
    end
    
    if drift_idx_spec(1,2)==1
        target_idx(1,width(target_idx)+1)=12;
    end
    
    if drift_idx_spec(1,3)==1
        target_idx(1,width(target_idx)+1)=13;
    end
    
    if drift_idx_spec(1,4)==1
        target_idx(1,width(target_idx)+1)=14;
    end
    
    ind_gas=find(ismember(gasnames,anchor_gas));
    ind_plateau_check=strcmp(binref.gastype,anchor_gas);
    
    ind_plateau_gas=[];
    %ind_plateau_gas=NaN(nnz(ind_plateau_check),1); %nnz = number of non-zero elements
    
    for i=1:height(ind_plateau_check)
        if ind_plateau_check(i,1)==1     
            ind_plateau_gas(height(ind_plateau_gas)+1,1)=i;
        end
    end
    
    %% check if CH4 data file was uploaded and modify target_idx accordingly

    if bindata.data_ch4_idx==0
        target_idx = target_idx(find(target_idx~=12 & target_idx~=13 & target_idx~=14));
        fprintf('Warning: No CH4,CO,CO2 data available\n');
    end
    
    
    %% interpolate reference values
    
    binref_intpl.(gasnames{1,ind_gas})=NaN(height(binref.mean),width(binref.mean));
    
    for k=1:height(bindata.boundaries_idx)-1
        if k > ind_plateau_gas(1,1) && k < ind_plateau_gas(end,1)
            if bindata.flag(k,1) ~= gastypeflag(1,ind_gas)
                ind_check_min=max(ind_plateau_gas(ind_plateau_gas<k));
                ind_check_max=min(ind_plateau_gas(ind_plateau_gas>k));
                deltapar=binref.mean(ind_check_max,:)-binref.mean(ind_check_min,:);
                slope=deltapar./deltapar(1,1);
                tmin=binref.mean(ind_check_min,1);
                parmin=binref.mean(ind_check_min,:);
                binref_intpl.(gasnames{1,ind_gas})(k,:)=slope*(binref.mean(k,1)-tmin)+parmin;
            end
        end
    end
    
    for i=1:height(binref_intpl.(gasnames{1,ind_gas}))
        if binref_intpl.(gasnames{1,ind_gas})(i,:)==0
            binref_intpl.(gasnames{1,ind_gas})(i,:)=NaN; end
    end
    
    %% interpolate bin data
    
    bindata_intpl.all=NaN(height(bindata.all),width(bindata.all));
    
        for k=1:height(bindata.boundaries_idx)-1
              if k > ind_plateau_gas(1,1) && k < ind_plateau_gas(end,1)
                   if bindata.flag(k,1) ~= gastypeflag(1,ind_gas)                  
                        ind_check_min=max(ind_plateau_gas(ind_plateau_gas<k));
                        ind_check_max=min(ind_plateau_gas(ind_plateau_gas>k));
                        deltapar=binref.mean(ind_check_max,:)-binref.mean(ind_check_min,:);
                        slope=deltapar./deltapar(1,1);
                        tmin=binref.mean(ind_check_min,1);
                        parmin=binref.mean(ind_check_min,:);
                        for j=1:width(bindata.all)        
                            bindata_intpl.all(bindata.boundaries_idx(k,1)+1:bindata.boundaries_idx(k+1,1),j)=slope(1,j)*(bindata.all(bindata.boundaries_idx(k,1)+1:bindata.boundaries_idx(k+1,1),1)-tmin(1,1))+parmin(1,j);
                        end          
                   end
              end
        end
    
     %% calculate drift correction ref values
     
     binref_intpl.driftcorr=NaN(height(binref.mean),width(binref.mean));
     
     for k=1:height(bindata.boundaries_idx)-1
          binref_intpl.driftcorr(k,1)=binref.mean(k,1);
         if bindata.flag(k,1) == gastypeflag(1,ind_gas)
             binref_intpl.driftcorr(k,2:end)=binref.mean(k,2:end)-binref.(gasnames{1,ind_gas})(1,2:end);
         end
         if bindata.flag(k,1) ~= gastypeflag(1,ind_gas)
             binref_intpl.driftcorr(k,2:end)=binref_intpl.(gasnames{1,ind_gas})(k,2:end)-binref.(gasnames{1,ind_gas})(1,2:end);
         end   
     end
      
     %% calculate drift correction for individual points
     
      bindata_intpl.driftcorr=NaN(height(bindata.all),width(bindata.all));
      bindata_intpl.driftcorr(:,1)=bindata.all(:,1);
      
         for k=1:height(bindata.boundaries_idx)-1    
            for i=1:width(gastypeflag)
                if bindata.flag(k,1) == gastypeflag(1,ind_gas)
                    bindata_intpl.driftcorr(bindata.boundaries_idx(k,1)+1:bindata.boundaries_idx(k+1,1),2:end)=repmat(binref_intpl.driftcorr(k,2:end),width(bindata.boundaries_idx(k,1)+1:bindata.boundaries_idx(k+1,1)),1);
                end
                if bindata.flag(k,1) ~= gastypeflag(1,ind_gas)
                    bindata_intpl.driftcorr(bindata.boundaries_idx(k,1)+1:bindata.boundaries_idx(k+1,1),2:end)=bindata_intpl.all(bindata.boundaries_idx(k,1)+1:bindata.boundaries_idx(k+1,1),2:end)-binref.(gasnames{1,ind_gas})(1,2:end);
                end
            end
        end
      
    %% plot interpolation (test figure)
    
     if width(target_idx) <=2
        nn=2;
     end
     if width(target_idx) ==3
        nn=3;
     end
      if width(target_idx) >=4
        nn=4;
     end
    if plot_requested
        figure;
        for z=1:width(target_idx)    
            subplot(nn,2,2*z-1);
            scatter(bindata.(gasnames{1,ind_gas})(:,1),bindata.(gasnames{1,ind_gas})(:,target_idx(1,z)),10);
            hold on;
%             for j=2:height(data.boundaries_idx)-1
%                 xline(data.all(data.boundaries_idx(j,1),2),'--k');
%             end
            %xline([data.all(data.boundaries_idx(1:height(data.boundaries_idx),1),1)],'--k');        
            errorbar(binref.mean(ind_plateau_gas,1),binref.mean(ind_plateau_gas,target_idx(1,z)),binref.std(ind_plateau_gas,target_idx(1,z)),'marker','o','linestyle','none','color','k','markeredgecolor','k','markerfacecolor','k','linewidth',1.5);
            scatter(bindata_intpl.all(:,1),bindata_intpl.all(:,target_idx(1,z)),10,'markeredgecolor',[0.7 0.7 0.7]);
            scatter(binref_intpl.(gasnames{1,ind_gas})(:,1),binref_intpl.(gasnames{1,ind_gas})(:,target_idx(1,z)),'k','marker','o','markeredgecolor','k','markerfacecolor','w','linewidth',1.5);
            grid on;
            box on;
            xlim([0 bindata.(gasnames{1,ind_gas})(end,1)]);
            ylabel({sprintf('%s',bindata.varlabels{1,target_idx(1,z)})},'fontsize',12);
            legend(sprintf('%s',anchor_gas),'location','northeast','fontsize',10);
            if z==1
                title('Reference dataset interpolation','fontsize',12); end      
            
            subplot(nn,2,2*z);
            hold on;
%             for j=2:height(data.boundaries_idx)-1
%                 xline(data.all(data.boundaries_idx(j,1),2),'--k');
%             end
            %xline([data.all(data.boundaries_idx(1:height(data.boundaries_idx),1),1)],'--k');
            yline(0,'--r');
            scatter(bindata_intpl.driftcorr(:,1),bindata_intpl.driftcorr(:,target_idx(1,z)),10,'markeredgecolor',[0.7 0.7 0.7]);
            errorbar(binref.mean(ind_plateau_gas,1),binref.mean(ind_plateau_gas,target_idx(1,z))-binref.(gasnames{1,ind_gas})(1,target_idx(1,z)),binref.std(ind_plateau_gas,target_idx(1,z)),'marker','o','linestyle','none','color','r','markeredgecolor','r','markerfacecolor','k','linewidth',1.5);
            plot(binref_intpl.driftcorr(:,1),binref_intpl.driftcorr(:,target_idx(1,z)),'-r','marker','o','markeredgecolor','r','markerfacecolor','r','linewidth',1.5);
            scatter(binref_intpl.(gasnames{1,ind_gas})(:,1),binref_intpl.(gasnames{1,ind_gas})(:,target_idx(1,z))-binref.(gasnames{1,ind_gas})(1,target_idx(1,z)),'r','marker','o','markeredgecolor','r','markerfacecolor','w','linewidth',1.5);
            grid on;
            box on;
            xlim([0 bindata.(gasnames{1,ind_gas})(end,1)]);
            ylabel({sprintf('%s - mean',bindata.varlabels{1,target_idx(1,z)})},'fontsize',12);
            if z==1
                title('Drift correction','fontsize',12); end
        end
        
            x0 = 10;
            y0 = 50;
            ww = 900;
            hh = 700;
            set(gcf,'units','points','position',[x0,y0,ww,hh]);
            set(gcf,'Units','Inches');
            pos = get(gcf,'Position');
            set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3),pos(4)]);
        
        %print(gcf,'C:\Users\brsi\Desktop\temp9_corr_before_n2o','-dpng', '-r800');
    end
    
    %% plot data before correction
    
    if plot_requested
        plot_drift_corr_conc(data,bindata,binref,gasnames,target_idx,anchor_gas,ind_gas,ind_plateau_gas,'Before drift correction');
    end
    
    %% apply correction (to all data)
    
    for z=1:width(target_idx)    
        bindata.all(:,target_idx(1,z))=bindata.all(:,target_idx(1,z))-bindata_intpl.driftcorr(:,target_idx(1,z));
    end
    
    %% re-define gas types vectors
      for k=1:width(gasnames)
            bindata.(gasnames{1,k})=NaN(height(bindata.all),width(bindata.all));
            for i=1:height(bindata.all)
                 if bindata.all(i,11)==gastypeflag(1,k)
                      bindata.(gasnames{1,k})(i,:)=bindata.all(i,:); 
                 end
            end
      end
      
    %% Remove empty rows 
      for k=1:width(gasnames)
          bindata.(gasnames{1,k})=rmmissing(bindata.(gasnames{1,k}),1,'MinNumMissing',width(bindata.all));
      end
      
    %% recalculate ref conditions
    [ref, binref, bindata] = calc_ref_conditions(data,bindata,gasnames,gastypeflag);
    
    %% plot data after correction
    
    if plot_requested
        plot_drift_corr_conc(data,bindata,binref,gasnames,target_idx,anchor_gas,ind_gas,ind_plateau_gas,'After drift correction');
    end
    
    %%
    time_elapsed=toc; fprintf('apply_drift_corr_conc completed (execution time: %1.2f s)\n',time_elapsed); 

end

